<?php
/**
 * Router minimalis (tanpa namespace, tanpa composer).
 * Mendukung route GET/POST dengan pencocokan path sederhana (exact match).
 */
class Router
{
    /** @var array<array{method:string,pattern:string,handler:mixed}> */
    private $routes = [];

    public function get($pattern, $handler)  { $this->add('GET',  $pattern, $handler); }
    public function post($pattern, $handler) { $this->add('POST', $pattern, $handler); }

    private function add($method, $pattern, $handler)
    {
        $this->routes[] = [
            'method'  => strtoupper($method),
            'pattern' => $this->normalize($pattern),
            'handler' => $handler
        ];
    }

    /**
     * Dispatch berdasarkan PATH & METHOD.
     * Jika tidak ditemukan, kirim 404 JSON yang jelas.
     */
    public function dispatch($path, $method)
    {
        $path = $this->normalize($path);
        $method = strtoupper($method);

        foreach ($this->routes as $r) {
            if ($r['method'] === $method && $r['pattern'] === $path) {
                return $this->invoke($r['handler']);
            }
        }

        // 404 default
        if (!headers_sent()) {
            http_response_code(404);
            header('Content-Type: application/json; charset=utf-8');
        }
        echo json_encode(['success' => false, 'error' => 'Endpoint tidak ditemukan.'], JSON_UNESCAPED_UNICODE);
        return null;
    }

    /**
     * Panggil handler: callable atau "NamaController@metode".
     */
    private function invoke($handler)
    {
        if (is_callable($handler)) {
            return $handler();
        }

        if (is_string($handler)) {
            $parts = explode('@', $handler, 2);
            $class  = $parts[0] ?? '';
            $method = $parts[1] ?? '';

            if (!class_exists($class) || !method_exists($class, $method)) {
                throw new Exception('Handler tidak valid pada router.');
            }

            $controller = new $class();
            return call_user_func([$controller, $method]);
        }

        throw new Exception('Handler tidak valid pada router.');
    }

    /**
     * Normalisasi path:
     * - Trim leading/trailing slashes
     * - Decode URL
     * - Hilangkan trailing slash kosong
     */
    private function normalize($path)
    {
        $path = is_string($path) ? $path : '';
        $path = urldecode($path);
        $path = trim($path);
        $path = trim($path, '/');
        return $path;
    }
}
